<?php
// إعدادات البوت وتحميل الملفات الأساسية
$config = json_decode(file_get_contents('config.json'), true);
$bot_token = $config['bot_token'];
$admin_id = $config['admin_id'];

// دوال مساعدة
function sendMessage($chat_id, $text, $reply_markup = null) {
    global $bot_token;
    $data = [
        'chat_id' => $chat_id,
        'text' => $text,
        'parse_mode' => 'HTML'
    ];
    if ($reply_markup) {
        $data['reply_markup'] = json_encode($reply_markup);
    }
    file_get_contents("https://api.telegram.org/bot$bot_token/sendMessage?" . http_build_query($data));
}

function editMessage($chat_id, $message_id, $text, $reply_markup = null) {
    global $bot_token;
    $data = [
        'chat_id' => $chat_id,
        'message_id' => $message_id,
        'text' => $text,
        'parse_mode' => 'HTML'
    ];
    if ($reply_markup) {
        $data['reply_markup'] = json_encode($reply_markup);
    }
    file_get_contents("https://api.telegram.org/bot$bot_token/editMessageText?" . http_build_query($data));
}

function answerCallback($callback_query_id, $text) {
    global $bot_token;
    $data = [
        'callback_query_id' => $callback_query_id,
        'text' => $text
    ];
    file_get_contents("https://api.telegram.org/bot$bot_token/answerCallbackQuery?" . http_build_query($data));
}

function loadJson($file) {
    if (!file_exists($file)) return [];
    $data = file_get_contents($file);
    if ($data === false) return [];
    $json = json_decode($data, true);
    if (!is_array($json)) return [];
    return $json;
}

function saveJson($file, $data) {
    $fp = fopen($file, 'w');
    if (flock($fp, LOCK_EX)) {
        fwrite($fp, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
        fflush($fp);
        flock($fp, LOCK_UN);
    } else {
        // فشل القفل
        throw new Exception('تعذر حفظ البيانات!');
    }
    fclose($fp);
}

// دوال السماح بالدخول
function isAllowed($user_id) {
    global $admin_id;
    if ($user_id == $admin_id) return true;
    $allowed = loadJson('allowed_users.json');
    return in_array($user_id, $allowed);
}
function addAllowedUser($user_id) {
    $allowed = loadJson('allowed_users.json');
    if (!in_array($user_id, $allowed)) {
        $allowed[] = $user_id;
        saveJson('allowed_users.json', $allowed);
    }
}
function removeAllowedUser($user_id) {
    $allowed = loadJson('allowed_users.json');
    $allowed = array_filter($allowed, function($id) use ($user_id) { return $id != $user_id; });
    saveJson('allowed_users.json', array_values($allowed));
}

// دوال المستخدمين
function getUser($user_id) {
    $users = loadJson('users.json');
    foreach ($users as $user) {
        if ($user['id'] == $user_id) return $user;
    }
    return null;
}

function saveUser($user) {
    $users = loadJson('users.json');
    $found = false;
    foreach ($users as $i => $u) {
        if ($u['id'] == $user['id']) {
            $users[$i] = $user;
            $found = true;
            break;
        }
    }
    if (!$found) $users[] = $user;
    saveJson('users.json', $users);
}

function registerUser($user_id, $name) {
    if (!getUser($user_id)) {
        $user = [
            'id' => $user_id,
            'name' => $name,
            'balance' => 0,
            'orders' => []
        ];
        saveUser($user);
    }
}

// دوال الخدمات
function getServices() {
    return loadJson('services.json');
}

function getCategories() {
    $services = getServices();
    $cats = [];
    foreach ($services as $srv) {
        if (!in_array($srv['category'], $cats)) $cats[] = $srv['category'];
    }
    return $cats;
}

function getServiceById($id) {
    $services = getServices();
    foreach ($services as $srv) {
        if ($srv['id'] == $id) return $srv;
    }
    return null;
}

// دوال الطلبات
function addOrder($order) {
    $orders = loadJson('orders.json');
    $orders[] = $order;
    saveJson('orders.json', $orders);
}

function getOrder($order_id) {
    $orders = loadJson('orders.json');
    foreach ($orders as $order) {
        if ($order['id'] == $order_id) return $order;
    }
    return null;
}

function updateOrder($order) {
    $orders = loadJson('orders.json');
    foreach ($orders as &$o) {
        if ($o['id'] == $order['id']) {
            $o = $order;
            break;
        }
    }
    saveJson('orders.json', $orders);
}

// دوال المزودين
function getProviders() {
    return loadJson('providers.json');
}

function getProviderById($id) {
    $providers = getProviders();
    foreach ($providers as $p) {
        if ($p['id'] == $id) return $p;
    }
    return null;
}

function addProvider($provider) {
    $providers = getProviders();
    $providers[] = $provider;
    saveJson('providers.json', $providers);
}

// دوال التعامل مع n1 API
function n1ApiRequest($provider, $params) {
    $params['key'] = $provider['api_key'];
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $provider['api_url']);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($params));
    $result = curl_exec($ch);
    curl_close($ch);
    return json_decode($result, true);
}

function n1AddOrder($provider, $service, $link, $quantity) {
    $params = [
        'action' => 'add',
        'service' => $service,
        'link' => $link,
        'quantity' => $quantity
    ];
    return n1ApiRequest($provider, $params);
}

function n1OrderStatus($provider, $order_id) {
    $params = [
        'action' => 'status',
        'order' => $order_id
    ];
    return n1ApiRequest($provider, $params);
}

function n1GetServices($provider) {
    $params = [
        'action' => 'services'
    ];
    return n1ApiRequest($provider, $params);
}

function n1GetBalance($provider) {
    $params = [
        'action' => 'balance'
    ];
    return n1ApiRequest($provider, $params);
}

// استقبال التحديثات
$update = json_decode(file_get_contents('php://input'), true);
if (!$update) exit();

// التحقق من السماح بالدخول
$user_id = $update['message']['from']['id'] ?? $update['callback_query']['from']['id'] ?? null;
if ($user_id && !isAllowed($user_id)) {
    sendMessage($user_id, "❌ أنت غير مصرح لك باستخدام هذا البوت. تواصل مع الأدمن.");
    exit;
}

// تحديد نوع التحديث
$message = $update['message'] ?? null;
$callback = $update['callback_query'] ?? null;

// معالجة الرسائل الرئيسية
if ($message) {
    $name = $message['from']['first_name'] . (isset($message['from']['last_name']) ? ' ' . $message['from']['last_name'] : '');
    $text = $message['text'] ?? '';
    registerUser($user_id, $name);
    $user = getUser($user_id);

    // لوحة الأدمن لإدارة المستخدمين
    if ($user_id == $admin_id && $text == '/admin') {
        $reply = [
            'keyboard' => [
                [['text' => '➕ إضافة مستخدم'], ['text' => '➖ حذف مستخدم']],
                [['text' => '⬅️ رجوع']]
            ],
            'resize_keyboard' => true
        ];
        sendMessage($user_id, "لوحة تحكم الأدمن:", $reply);
        exit;
    }
    if ($user_id == $admin_id && $text == '➕ إضافة مستخدم') {
        sendMessage($user_id, "أرسل آيدي المستخدم الذي تريد السماح له:");
        file_put_contents("admin_step_$user_id.txt", 'add_user');
        exit;
    }
    if ($user_id == $admin_id && $text == '➖ حذف مستخدم') {
        sendMessage($user_id, "أرسل آيدي المستخدم الذي تريد حذفه:");
        file_put_contents("admin_step_$user_id.txt", 'del_user');
        exit;
    }
    if ($user_id == $admin_id && file_exists("admin_step_$user_id.txt")) {
        $step = file_get_contents("admin_step_$user_id.txt");
        if ($step == 'add_user') {
            $target = intval($text);
            addAllowedUser($target);
            sendMessage($user_id, "تمت إضافة المستخدم $target بنجاح.");
            unlink("admin_step_$user_id.txt");
            exit;
        }
        if ($step == 'del_user') {
            $target = intval($text);
            removeAllowedUser($target);
            sendMessage($user_id, "تم حذف المستخدم $target من قائمة المسموح لهم.");
            unlink("admin_step_$user_id.txt");
            exit;
        }
    }
    // واجهة المستخدم الرئيسية
    if ($text == '/start' || $text == '⬅️ رجوع') {
        $reply = [
            'keyboard' => [
                [['text' => '🛒 طلب خدمة']],
            ],
            'resize_keyboard' => true
        ];
        sendMessage($user_id, "مرحباً <b>$name</b>!\nاختر من القائمة:", $reply);
        file_put_contents("step_$user_id.txt", '');
        exit;
    }
    // طلب خدمة
    if ($text == '🛒 طلب خدمة') {
        $cats = getCategories();
        if (!$cats) {
            sendMessage($user_id, "لا توجد خدمات متاحة حالياً.");
            exit;
        }
        $btns = [];
        foreach ($cats as $cat) $btns[] = [['text' => $cat]];
        $btns[] = [['text' => '⬅️ رجوع']];
        $reply = ['keyboard' => $btns, 'resize_keyboard' => true];
        sendMessage($user_id, "اختر التصنيف:", $reply);
        file_put_contents("step_$user_id.txt", 'wait_category');
        exit;
    }
    // اختيار تصنيف خدمة
    $step = @file_get_contents("step_$user_id.txt");
    if ($step == 'wait_category' && in_array($text, getCategories())) {
        $srvs = getServices();
        $btns = [];
        foreach ($srvs as $srv) {
            if ($srv['category'] == $text) {
                $btns[] = [['text' => $srv['name'] . " | " . $srv['rate'] . "$/1000", 'callback_data' => 'srv_' . $srv['id']]];
            }
        }
        if (!$btns) {
            sendMessage($user_id, "لا توجد خدمات في هذا التصنيف.");
            exit;
        }
        $reply = ['inline_keyboard' => $btns];
        sendMessage($user_id, "اختر الخدمة:", $reply);
        file_put_contents("step_$user_id.txt", '');
        exit;
    }
    // إدخال الرابط
    if ($step && strpos($step, 'wait_link_') === 0) {
        $srv_id = intval(str_replace('wait_link_', '', $step));
        $srv = getServiceById($srv_id);
        $link = trim($text);
        sendMessage($user_id, "أدخل الكمية المطلوبة (الحد الأدنى: " . $srv['min'] . ", الحد الأقصى: " . $srv['max'] . "):");
        file_put_contents("step_$user_id.txt", 'wait_qty_' . $srv_id . '_' . base64_encode($link));
        exit;
    }
    // إدخال الكمية
    if ($step && strpos($step, 'wait_qty_') === 0) {
        $parts = explode('_', $step, 4);
        $srv_id = intval($parts[2]);
        $link = base64_decode($parts[3]);
        $srv = getServiceById($srv_id);
        $qty = intval($text);
        if ($qty < $srv['min'] || $qty > $srv['max']) {
            sendMessage($user_id, "الكمية خارج الحدود المسموحة.");
            exit;
        }
        $cost = round(($qty / 1000) * floatval($srv['rate']), 4);
        $summary = "ملخص طلبك:\nالخدمة: <b>" . $srv['name'] . "</b>\nالرابط: <code>$link</code>\nالكمية: <b>$qty</b>\nالسعر: <b>$cost</b> دولار\nأرسل (تأكيد) للمتابعة أو (إلغاء) لإلغاء الطلب.";
        sendMessage($user_id, $summary);
        file_put_contents("step_$user_id.txt", 'wait_confirm_' . $srv_id . '_' . base64_encode($link) . '_' . $qty . '_' . $cost);
        exit;
    }
    // تأكيد الطلب
    if ($step && strpos($step, 'wait_confirm_') === 0) {
        $parts = explode('_', $step, 6);
        $srv_id = intval($parts[2]);
        $link = base64_decode($parts[3]);
        $qty = intval($parts[4]);
        $cost = floatval($parts[5]);
        $srv = getServiceById($srv_id);
        if (strtolower($text) == 'تأكيد') {
            // إرسال الطلب للأدمن للموافقة
            $order_id = uniqid();
            $order = [
                'id' => $order_id,
                'user_id' => $user_id,
                'service_id' => $srv_id,
                'link' => $link,
                'qty' => $qty,
                'cost' => $cost,
                'status' => 'بانتظار موافقة الأدمن',
                'created_at' => time()
            ];
            addOrder($order);
            $admin_msg = "طلب جديد بانتظار الموافقة:\nالعميل: <code>$user_id</code>\nالخدمة: <b>" . $srv['name'] . "</b>\nالرابط: <code>$link</code>\nالكمية: <b>$qty</b>\nالسعر: <b>$cost</b> دولار\nرقم الطلب: <code>$order_id</code>";
            $reply = [
                'inline_keyboard' => [
                    [
                        ['text' => '✅ موافقة', 'callback_data' => 'approve_' . $order_id],
                        ['text' => '❌ رفض', 'callback_data' => 'reject_' . $order_id]
                    ]
                ]
            ];
            sendMessage($admin_id, $admin_msg, $reply);
            sendMessage($user_id, "تم إرسال طلبك للأدمن للموافقة. سيتم إعلامك بالنتيجة.");
            file_put_contents("step_$user_id.txt", '');
            exit;
        } elseif (strtolower($text) == 'إلغاء') {
            sendMessage($user_id, "تم إلغاء الطلب.");
            file_put_contents("step_$user_id.txt", '');
            exit;
        } else {
            sendMessage($user_id, "يرجى كتابة (تأكيد) أو (إلغاء)");
            exit;
        }
    }
}

// معالجة callback_query (اختيار خدمة أو موافقة/رفض الأدمن)
if ($callback) {
    $user_id = $callback['from']['id'];
    $data = $callback['data'];
    $message_id = $callback['message']['message_id'];
    // اختيار خدمة
    if (strpos($data, 'srv_') === 0) {
        $srv_id = intval(str_replace('srv_', '', $data));
        $srv = getServiceById($srv_id);
        if (!$srv) {
            answerCallback($callback['id'], 'الخدمة غير موجودة!');
            exit;
        }
        editMessage($user_id, $message_id, "أرسل الرابط المطلوب للخدمة:");
        file_put_contents("step_$user_id.txt", 'wait_link_' . $srv_id);
        answerCallback($callback['id'], 'أرسل الرابط الآن');
        exit;
    }
    // موافقة الأدمن
    if (strpos($data, 'approve_') === 0 && $user_id == $admin_id) {
        $order_id = str_replace('approve_', '', $data);
        $orders = loadJson('orders.json');
        foreach ($orders as &$order) {
            if ($order['id'] == $order_id && $order['status'] == 'بانتظار موافقة الأدمن') {
                $srv = getServiceById($order['service_id']);
                $provider = getProviderById($srv['provider_id']);
                $api_res = n1AddOrder($provider, $srv['api_service_id'], $order['link'], $order['qty']);
                if (isset($api_res['order'])) {
                    $order['status'] = 'تم التنفيذ';
                    $order['api_order_id'] = $api_res['order'];
                    saveJson('orders.json', $orders);
                    editMessage($admin_id, $message_id, "تم تنفيذ الطلب وإبلاغ المستخدم.");
                    sendMessage($order['user_id'], "تمت الموافقة على طلبك وتم إرساله للمزود. رقم الطلب: <code>" . $api_res['order'] . "</code>");
                } else {
                    $order['status'] = 'فشل التنفيذ';
                    saveJson('orders.json', $orders);
                    editMessage($admin_id, $message_id, "فشل تنفيذ الطلب عند المزود.");
                    sendMessage($order['user_id'], "عذراً، حدث خطأ أثناء تنفيذ طلبك عند المزود. يرجى التواصل مع الأدمن.");
                }
                exit;
            }
        }
        answerCallback($callback['id'], 'لم يتم العثور على الطلب أو تم تنفيذه مسبقاً.');
        exit;
    }
    // رفض الأدمن
    if (strpos($data, 'reject_') === 0 && $user_id == $admin_id) {
        $order_id = str_replace('reject_', '', $data);
        $orders = loadJson('orders.json');
        foreach ($orders as &$order) {
            if ($order['id'] == $order_id && $order['status'] == 'بانتظار موافقة الأدمن') {
                $order['status'] = 'مرفوض من الأدمن';
                saveJson('orders.json', $orders);
                editMessage($admin_id, $message_id, "تم رفض الطلب.");
                sendMessage($order['user_id'], "تم رفض طلبك من قبل الأدمن.");
                exit;
            }
        }
        answerCallback($callback['id'], 'لم يتم العثور على الطلب أو تم تنفيذه مسبقاً.');
        exit;
    }
}

// ... سيتم إكمال باقي الوظائف لاحقاً ... 